#!/usr/bin/env node
/**
 * Shai-Hulud Worm Security Scanner (Multi-Project Edition + HTML Dashboard + PDF)
 * v6.1 — October 2025
 *
 * Recursively scans all subprojects for indicators of the Shai-Hulud npm worm
 * and other suspicious behaviors. Generates:
 *  - Individual reports
 *  - Combined JSON summary
 *  - HTML dashboard
 *  - Adds automatic PDF dashboard export
 */

import fs from "fs";
import path from "path";
import https from "https";
import child_process from "child_process";

let puppeteer = null;
try {
  puppeteer = await import("puppeteer");
} catch {
  puppeteer = null;
}

const SCRIPT_DIR = process.cwd();
const REPORT_DIR = path.join(SCRIPT_DIR, "shaihulud-reports");
const CACHE_FILE = path.join(SCRIPT_DIR, ".shaihulud-cache.json");
const FEEDS = ["https://registry.npmjs.org/-/v1/security/advisories?per_page=250"];
const CUTOFF_DATE = new Date("2025-09-01T00:00:00Z");

const log = (msg) => console.log(`[${new Date().toISOString()}] ${msg}`);
const run = (cmd) => {
  try {
    return child_process.execSync(cmd, { encoding: "utf8", stdio: "pipe" }).trim();
  } catch {
    return "";
  }
};

function fetchJSON(url) {
  return new Promise((resolve, reject) => {
    https
      .get(url, { timeout: 15000 }, (res) => {
        if (res.statusCode !== 200) return reject(new Error(`HTTP ${res.statusCode}`));
        let data = "";
        res.on("data", (chunk) => (data += chunk));
        res.on("end", () => {
          try {
            resolve(JSON.parse(data));
          } catch (err) {
            reject(err);
          }
        });
      })
      .on("error", reject);
  });
}

async function getCompromisedPackages() {
  for (const feed of FEEDS) {
    try {
      log(`Fetching IoC feed from ${feed}...`);
      const json = await fetchJSON(feed);
      const compromised = json.objects
        .filter(
          (a) =>
            /shai[-_ ]?hulud/i.test(a.title) ||
            /supply.?chain/i.test(a.title) ||
            /npm.?worm/i.test(a.overview)
        )
        .map((a) => a.module_name)
        .filter(Boolean);
      if (compromised.length > 0) {
        fs.writeFileSync(
          CACHE_FILE,
          JSON.stringify({ updated: Date.now(), list: compromised }, null, 2)
        );
        log(`✅ Loaded ${compromised.length} compromised packages.`);
        return compromised;
      }
    } catch (e) {
      log(`⚠️  Failed to fetch from ${feed}: ${e.message}`);
    }
  }
  if (fs.existsSync(CACHE_FILE)) {
    const cache = JSON.parse(fs.readFileSync(CACHE_FILE, "utf8"));
    log("Using cached IoC list...");
    return cache.list || [];
  }
  log("⚠️  No IoC list available (offline + no cache). Using fallback.");
  return ["lodash-shaihulud", "npm-run-shell"];
}

function isAfterWormWindow(dateStr) {
  const d = new Date(dateStr);
  return d > CUTOFF_DATE;
}

function calculateRiskScore(result) {
  let score = 0;
  score += result.dependencies.found.length > 0 ? 60 : 0;
  score += result.dependencies.recent.length * 5;
  score += result.scripts.flagged.length > 0 ? 10 : 0;
  score += result.workflows.suspicious.length > 0 ? 10 : 0;
  score += Math.min(result.secrets.hits.length * 5, 25);
  if (score > 100) score = 100;
  let level = "Low";
  if (score >= 61) level = "High";
  else if (score >= 26) level = "Medium";
  return { score, level };
}

function checkDependencies(compromised) {
  if (!fs.existsSync("package-lock.json")) return { found: [], recent: [] };
  const packageLock = JSON.parse(fs.readFileSync("package-lock.json", "utf8"));
  const found = [];
  const recent = [];
  function recurse(deps) {
    for (const [name, info] of Object.entries(deps)) {
      if (compromised.includes(name)) found.push({ name, version: info.version });
      if (info.resolved && isAfterWormWindow(info.resolved.split("/").pop()))
        recent.push({ name, version: info.version });
      if (info.dependencies) recurse(info.dependencies);
    }
  }
  if (packageLock.dependencies) recurse(packageLock.dependencies);
  return { found, recent };
}

function checkScripts() {
  if (!fs.existsSync("package.json")) return { flagged: [] };
  const pkg = JSON.parse(fs.readFileSync("package.json", "utf8"));
  const flagged = [];
  for (const [script, cmd] of Object.entries(pkg.scripts || {})) {
    if (script.includes("postinstall") || /curl|wget|base64|eval|bash/i.test(cmd)) {
      flagged.push({ script, cmd });
    }
  }
  return { flagged };
}

function checkWorkflows() {
  const dir = ".github/workflows";
  if (!fs.existsSync(dir)) return { suspicious: [] };
  const suspicious = [];
  for (const file of fs.readdirSync(dir)) {
    const content = fs.readFileSync(path.join(dir, file), "utf8");
    if (/shai[-_]?hulud/i.test(content) || /curl .*githubusercontent/i.test(content)) {
      suspicious.push(file);
    }
  }
  return { suspicious };
}

function checkSecrets() {
  let files = [];
  try {
    files = run("git ls-files").split("\n");
  } catch {
    return { hits: [] };
  }
  const regex =
    /(gh[pousr]_[A-Za-z0-9]{36,})|(npm_[A-Za-z0-9]{36,})|("aws_access_key_id"\s*:\s*")/i;
  const hits = [];
  for (const file of files) {
    if (!fs.existsSync(file) || fs.statSync(file).size > 2_000_000) continue;
    const content = fs.readFileSync(file, "utf8");
    if (regex.test(content)) hits.push(file);
  }
  return { hits };
}

function writeReport(projectPath, report) {
  if (!fs.existsSync(REPORT_DIR)) fs.mkdirSync(REPORT_DIR);
  const ts = new Date().toISOString().replace(/[-:T]/g, "").split(".")[0];
  const file = path.join(REPORT_DIR, `report-${path.basename(projectPath)}-${ts}.json`);
  fs.writeFileSync(file, JSON.stringify(report, null, 2));
  return file;
}

function writeSummaryReport(summary) {
  if (!fs.existsSync(REPORT_DIR)) fs.mkdirSync(REPORT_DIR);
  const file = path.join(
    REPORT_DIR,
    `summary-${new Date().toISOString().replace(/[-:T]/g, "").split(".")[0]}.json`
  );
  fs.writeFileSync(file, JSON.stringify(summary, null, 2));
  return file;
}

function findProjects(baseDir) {
  const projects = [];
  const entries = fs.readdirSync(baseDir, { withFileTypes: true });
  for (const entry of entries) {
    const full = path.join(baseDir, entry.name);
    if (entry.isDirectory()) {
      if (fs.existsSync(path.join(full, "package.json"))) {
        projects.push(full);
      } else {
        projects.push(...findProjects(full));
      }
    }
  }
  return projects;
}

(async function main() {
  const baseDir = process.argv[2] || ".";
  log(`=== Shai-Hulud Scanner v6.2 ===`);
  log(`Scanning projects under: ${baseDir}`);

  const compromised = await getCompromisedPackages();
  const projects = findProjects(baseDir);

  if (projects.length === 0) {
    log("❌ No projects found.");
    process.exit(0);
  }

  const summary = {
    timestamp: new Date().toISOString(),
    baseDir,
    totalProjects: projects.length,
    results: [],
  };

  for (const project of projects) {
    log(`\n🔍 Scanning: ${project}`);
    try {
      const cwd = process.cwd();
      process.chdir(project);
      const result = {
        project,
        dependencies: checkDependencies(compromised),
        scripts: checkScripts(),
        workflows: checkWorkflows(),
        secrets: checkSecrets(),
      };
      process.chdir(cwd);
      const risk = calculateRiskScore(result);
      result.risk = risk;
      const reportFile = writeReport(project, result);
      summary.results.push({ project, reportFile, ...result });
      log(`✅ ${path.basename(project)} — Risk ${risk.score} (${risk.level})`);
    } catch (err) {
      log(`⚠️  Error scanning ${project}: ${err.message}`);
    }
  }

  writeSummaryReport(summary);
  const dashboardPath = writeDashboard(summary);


  // Handle PDF export safely
  if (puppeteer) {
    log(`📊 Generating PDF version...`);
    await exportPDF(dashboardPath);
  } else {
    log(`⚠️ PDF generation skipped — Puppeteer not installed.`);
    log(`   To enable PDF export, run: npm install puppeteer`);
  }

  log(`\n✅ Done.`);
  log(`📁 Reports saved in: ${REPORT_DIR}`);
})();

function writeDashboard(summary) {
  if (!fs.existsSync(REPORT_DIR)) fs.mkdirSync(REPORT_DIR);
  const file = path.join(REPORT_DIR, "dashboard-shaihulud.html");
  const low = summary.results.filter((r) => r.risk.level === "Low").length;
  const med = summary.results.filter((r) => r.risk.level === "Medium").length;
  const high = summary.results.filter((r) => r.risk.level === "High").length;
  const avg = Math.round(
    summary.results.reduce((a, b) => a + b.risk.score, 0) / summary.results.length
  );

  const html = `
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>Shai-Hulud Security Dashboard</title>
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<style>
body { font-family: system-ui, sans-serif; background:#f9fafb; color:#111; margin:40px; }
h1 { font-size:1.8em; }
table { width:100%; border-collapse:collapse; margin-top:20px; }
th, td { padding:10px; border-bottom:1px solid #ddd; text-align:left; }
th { background:#f3f4f6; }
tr:hover { background:#f1f5f9; }
.risk-low { color:#16a34a; }
.risk-medium { color:#f59e0b; }
.risk-high { color:#dc2626; }
.bar { height:12px; border-radius:6px; }
.container { max-width:1200px; margin:auto; }
.chart-container { width:500px; height:300px; margin:20px auto; }
</style>
</head>
<body>
<div class="container">
<h1>🐛 Shai-Hulud Worm Security Dashboard</h1>
<p><strong>Total Projects:</strong> ${summary.totalProjects}<br>
<strong>Average Risk:</strong> ${avg}/100</p>
<div class="chart-container"><canvas id="chart"></canvas></div>
<table>
<thead><tr><th>Project</th><th>Risk Score</th><th>Level</th><th>Compromised</th><th>Scripts</th><th>Workflows</th><th>Secrets</th></tr></thead>
<tbody>
${summary.results
  .map(
    (r) => `
<tr>
<td>${r.project}</td>
<td><div class="bar" style="width:${r.risk.score}%; background:${
      r.risk.level === "High"
        ? "#dc2626"
        : r.risk.level === "Medium"
        ? "#f59e0b"
        : "#16a34a"
    };"></div>${r.risk.score}</td>
<td class="risk-${r.risk.level.toLowerCase()}">${r.risk.level}</td>
<td>${r.dependencies.found.length}</td>
<td>${r.scripts.flagged.length}</td>
<td>${r.workflows.suspicious.length}</td>
<td>${r.secrets.hits.length}</td>
</tr>`
  )
  .join("")}
</tbody>
</table>
<script>
const ctx = document.getElementById('chart');
new Chart(ctx, {
  type: 'pie',
  data: { labels: ['Low', 'Medium', 'High'], datasets: [{ data: [${low}, ${med}, ${high}], backgroundColor: ['#16a34a', '#f59e0b', '#dc2626'] }] },
  options: { plugins: { legend: { position: 'bottom' } } }
});
</script>
</div>
</body>
</html>
`;
  fs.writeFileSync(file, html);
  return file;
}

async function exportPDF(htmlPath) {
  const pdfPath = htmlPath.replace(/\.html$/, ".pdf");
  const browser = await puppeteer.launch({ headless: "new" });
  const page = await browser.newPage();
  await page.goto(`file://${htmlPath}`, { waitUntil: "networkidle0" });
  await page.pdf({ path: pdfPath, format: "A4", printBackground: true });
  await browser.close();
  log(`📄 PDF generated: ${pdfPath}`);
}

